<?php
// This file is part of the Checklist plugin for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Extra upgrade steps
 * @copyright Davo Smith <moodle@davosmith.co.uk>
 * @package mod_checklist
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/**
 * Extra upgrade steps.
 * @param int $oldversion
 * @return bool
 */
function xmldb_checklist_upgrade($oldversion = 0) {
    global $DB, $OUTPUT, $CFG;

    $dbman = $DB->get_manager();
    $result = true;

    if ($result && $oldversion < 2010022500) {
        // Adjust (currently unused) 'teachermark' fields to be 0 when unmarked, not 2.
        $sql = 'UPDATE {checklist_check} ';
        $sql .= 'SET teachermark=0 ';
        $sql .= 'WHERE teachermark=2';
        $DB->execute($sql);

        upgrade_mod_savepoint($result, 2010022500, 'checklist');
    }

    if ($result && $oldversion < 2010022800) {
        // All checklists created before this point were 'student only' checklists
        // Update the default & previously created checklists to reflect this.

        $sql = 'UPDATE {checklist} ';
        $sql .= 'SET teacheredit=0 ';
        $sql .= 'WHERE teacheredit=2';
        $DB->execute($sql);

        $table = new xmldb_table('checklist');
        $field = new xmldb_field('teacheredit', XMLDB_TYPE_INTEGER, '4', null, null, null, '0', 'useritemsallowed');
        $dbman->change_field_type($table, $field);

        // Checklist savepoint reached.
        upgrade_mod_savepoint($result, 2010022800, 'checklist');
    }

    if ($result && $oldversion < 2010031600) {
        $OUTPUT->notification('Processing checklist grades, this may take a while if there are many checklists...',
                              'notifysuccess');

        require_once(dirname(dirname(__FILE__)).'/lib.php');

        // Too much debug output.
        $olddebug = $DB->get_debug();
        $DB->set_debug(false);
        checklist_update_all_grades();
        $DB->set_debug($olddebug);

        // Checklist savepoint reached.
        upgrade_mod_savepoint($result, 2010031600, 'checklist');
    }

    if ($result && $oldversion < 2010041800) {
        $table = new xmldb_table('checklist_item');
        $field = new xmldb_field('duetime', XMLDB_TYPE_INTEGER, '10', null, null, null, '0', 'itemoptional');

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint($result, 2010041800, 'checklist');
    }

    if ($result && $oldversion < 2010041801) {
        $table = new xmldb_table('checklist');
        $field = new xmldb_field('duedatesoncalendar', XMLDB_TYPE_INTEGER, '4', null, null, null, '0', 'theme');

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint($result, 2010041801, 'checklist');
    }

    if ($result && $oldversion < 2010041900) {

        // Define field eventid to be added to checklist_item.
        $table = new xmldb_table('checklist_item');
        $field = new xmldb_field('eventid', XMLDB_TYPE_INTEGER, '10', null, null, null, '0', 'duetime');

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint($result, 2010041900, 'checklist');
    }

    if ($result && $oldversion < 2010050100) {

        // Define field teachercomments to be added to checklist.
        $table = new xmldb_table('checklist');
        $field = new xmldb_field('teachercomments', XMLDB_TYPE_INTEGER, '4', null, null, null, '1',
                                 'duedatesoncalendar');

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Define table checklist_comment to be created.
        $table = new xmldb_table('checklist_comment');

        // Adding fields to table checklist_comment.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('itemid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('userid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('commentby', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        $table->add_field('text', XMLDB_TYPE_TEXT, 'medium', null, null, null, null);

        // Adding keys to table checklist_comment.
        $table->add_key('primary', XMLDB_KEY_PRIMARY, ['id']);

        // Adding indexes to table checklist_comment.
        $table->add_index('checklist_item_user', XMLDB_INDEX_UNIQUE, ['itemid', 'userid']);

        // Conditionally launch create table for checklist_comment.
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint($result, 2010050100, 'checklist');
    }

    if ($result && $oldversion < 2010091003) {
        $table = new xmldb_table('checklist_item');
        $field = new xmldb_field('colour', XMLDB_TYPE_CHAR, '15', null, XMLDB_NOTNULL, null, 'black');

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_mod_savepoint($result, 2010091003, 'checklist');
    }

    if ($result && $oldversion < 2010102703) {
        $table = new xmldb_table('checklist');
        $field = new xmldb_field('maxgrade', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, '100');

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_mod_savepoint($result, 2010102703, 'checklist');
    }

    if ($result && $oldversion < 2010112000) {
        $table = new xmldb_table('checklist');
        $field = new xmldb_field('autopopulate', XMLDB_TYPE_INTEGER, '4', null, null, null, '0');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }
        $field = new xmldb_field('autoupdate', XMLDB_TYPE_INTEGER, '4', null, null, null, '1');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $table = new xmldb_table('checklist_item');
        $field = new xmldb_field('moduleid', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $table->add_index('item_module', XMLDB_INDEX_NOTUNIQUE, ['moduleid']);

        upgrade_mod_savepoint($result, 2010112000, 'checklist');
    }

    if ($result && $oldversion < 2010113000) {
        $table = new xmldb_table('checklist');
        $field = new xmldb_field('completionpercent', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_mod_savepoint($result, 2010113000, 'checklist');
    }

    if ($result && $oldversion < 2011021600) {
        // I really should not have to update the 'cron' field manually.
        $chkmod = $DB->get_record('modules', ['name' => 'checklist']);
        if ($chkmod) {
            $chkmodupd = new stdClass;
            $chkmodupd->id = $chkmod->id;
            $chkmodupd->cron = 60;
            $DB->update_record('modules', $chkmodupd);
        }
    }

    if ($result && $oldversion < 2011021900) {
        $table = new xmldb_table('checklist_item');
        $field = new xmldb_field('hidden', XMLDB_TYPE_INTEGER, '4', null, XMLDB_NOTNULL, null, '0');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Switch alll 'hidden headings' to being headings & hidden.
        $sql = 'UPDATE {checklist_item} ';
        $sql .= 'SET hidden=1, itemoptional=2 ';
        $sql .= 'WHERE itemoptional=4';
        $DB->execute($sql);

        // Switch all 'hidden items' to being required items & hidden.
        $sql = 'UPDATE {checklist_item} ';
        $sql .= 'SET hidden=1, itemoptional=0 ';
        $sql .= 'WHERE itemoptional=3';
        $DB->execute($sql);

        upgrade_mod_savepoint($result, 2011021900, 'checklist');
    }

    if ($result && $oldversion < 2011022700) {
        $table = new xmldb_table('checklist_item');
        $field = new xmldb_field('grouping', XMLDB_TYPE_INTEGER, '10', null, null, null, '0', 'hidden');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_mod_savepoint($result, 2011022700, 'checklist');
    }

    if ($result && $oldversion < 2011052901) {
        $table = new xmldb_table('checklist');
        $field = new xmldb_field('emailoncomplete', XMLDB_TYPE_INTEGER, '4', null, null, null, '0');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_mod_savepoint($result, 2011052901, 'checklist');
    }

    if ($result && $oldversion < 2011082001) {
        $table = new xmldb_table('checklist');
        $field = new xmldb_field('lockteachermarks', XMLDB_TYPE_INTEGER, '4', null, null, null, '0');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_mod_savepoint($result, 2011082001, 'checklist');
    }

    if ($oldversion < 2012092002) {

        // Define field teacherid to be added to checklist_check.
        $table = new xmldb_table('checklist_check');
        $field = new xmldb_field('teacherid', XMLDB_TYPE_INTEGER, '10', null, null, null, null, 'teachertimestamp');

        // Conditionally launch add field teacherid.
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint(true, 2012092002, 'checklist');
    }

    if ($oldversion < 2016090902) {

        $table = new xmldb_table('checklist_item');

        // Define field linkcourseid to be added to checklist_item.
        $field = new xmldb_field('linkcourseid', XMLDB_TYPE_INTEGER, '10', null, null, null, null, 'grouping');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);

            // Define key linkcourseid (foreign) to be added to checklist_item.
            $key = new xmldb_key('linkcourseid', XMLDB_KEY_FOREIGN, ['linkcourseid'], 'course', ['id']);
            $dbman->add_key($table, $key);
        }

        // Define field linkurl to be added to checklist_item.
        $field = new xmldb_field('linkurl', XMLDB_TYPE_TEXT, null, null, null, null, null, 'linkcourseid');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint(true, 2016090902, 'checklist');
    }

    if ($oldversion < 2018051500) {
        // This version includes the extended privacy API only found in M3.4.6, M3.5.3 and M3.6+.
        if ($CFG->version > 2018051700 && $CFG->version < 2018051703) {
            // Main version.php takes care of Moodle below 3.4.6.
            die('You must upgrade to Moodle 3.5.3 (or above) before upgrading to this version of mod_checklist');
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint(true, 2018051500, 'checklist');
    }

    if ($oldversion < 2019061900) {
        $table = new xmldb_table('checklist_item');
        $field = new xmldb_field('grouping', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');

        // Rename field grouping as "GROUPING" has become a reserved keyword in MySQL 8.
        if ($dbman->field_exists($table, $field)) {
            $dbman->rename_field($table, $field, 'groupingid');
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint(true, 2019061900, 'checklist');
    }

    if ($oldversion < 2020012900) {
        $table = new xmldb_table('checklist_item');
        $field = new xmldb_field('openlinkinnewwindow', XMLDB_TYPE_INTEGER, '1', null, XMLDB_NOTNULL, null, 0,
                                 'linkurl');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }
        // Checklist savepoint reached.
        upgrade_mod_savepoint(true, 2020012900, 'checklist');
    }

    if ($oldversion < 2020061500) {

        // Define field completionpercenttype to be added to checklist.
        $table = new xmldb_table('checklist');
        $field = new xmldb_field('completionpercenttype', XMLDB_TYPE_CHAR, '8', null, XMLDB_NOTNULL, null, 'percent',
                                 'completionpercent');

        // Conditionally launch add field completionpercenttype.
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint(true, 2020061500, 'checklist');
    }

    if ($oldversion < 2021091804) {

        // Add config field to checklist.
        $table = new xmldb_table('checklist');
        $field = new xmldb_field('studentcomments', XMLDB_TYPE_INTEGER, '4', null, null, null, '0', 'useritemsallowed');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Add table.
        $table = new xmldb_table('checklist_comment_student');
        $table->add_field('id', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, XMLDB_SEQUENCE);
        $table->add_field('itemid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL);
        $table->add_field('usermodified', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL);
        $table->add_field('text', XMLDB_TYPE_TEXT);
        $table->add_field('timecreated', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL);
        $table->add_field('timemodified', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL);

        // Adding keys to table checklist_comment_student.
        $table->add_key('primary', XMLDB_KEY_PRIMARY, ['id']);

        // Adding indexes to table checklist_comment_student.
        $table->add_index('checklist_item_student', XMLDB_INDEX_UNIQUE, ['itemid', 'usermodified']);

        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint(true, 2021091804, 'checklist');
    }

    if ($oldversion < 2022052800) {

        // Define table checklist_comp_notification to be created.
        $table = new xmldb_table('checklist_comp_notification');

        // Adding fields to table checklist_comp_notification.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('checklistid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, null);
        $table->add_field('userid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, null);
        $table->add_field('iscomplete', XMLDB_TYPE_INTEGER, '2', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('timenotified', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, '0');

        // Adding keys to table checklist_comp_notification.
        $table->add_key('primary', XMLDB_KEY_PRIMARY, ['id']);
        $table->add_key('checklistid', XMLDB_KEY_FOREIGN, ['checklistid'], 'checklist', ['id']);
        $table->add_key('userid', XMLDB_KEY_FOREIGN, ['userid'], 'user', ['id']);

        // Adding indexes to table checklist_comp_notification.
        $table->add_index('checklistid_userid', XMLDB_INDEX_UNIQUE, ['checklistid', 'userid']);

        // Conditionally launch create table for checklist_comp_notification.
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Checklist savepoint reached.
        upgrade_mod_savepoint(true, 2022052800, 'checklist');
    }

    if ($oldversion < 2022052801) {
        \core\task\manager::queue_adhoc_task(new \mod_checklist\task\update_all_grades());
        upgrade_mod_savepoint(true, 2022052801, 'checklist');
    }

    return $result;
}
